<?php
header('Content-Type: application/json; charset=utf-8');
session_start();
require 'config.php';

if (empty($_SESSION['id'])) {
    echo json_encode(['status'=>'error','message'=>'Unauthorized']);
    exit;
}
$student_ids = $_POST['student_ids'] ?? [];
$dept_id     = isset($_POST['dept_id']) ? (int)$_POST['dept_id'] : 0;
$class_id    = isset($_POST['class_id']) ? (int)$_POST['class_id'] : 0;
$section_id    = isset($_POST['section_id']) ? (int)$_POST['section_id'] : 0;

if (!is_array($student_ids) || count($student_ids) === 0) {
    echo json_encode(['status'=>'error','message'=>'No students selected']);
    exit;
}
$selectedCnt = count($student_ids);

if (!$dept_id || !$class_id) {
    echo json_encode(['status'=>'error','message'=>'Please select Class and Department']);
    exit;
}

$student_ids = array_values(array_map('intval', $student_ids));
$placeholders = implode(',', array_fill(0, count($student_ids), '?'));

$getTokensStmt = $pdo->prepare("SELECT DISTINCT school_token FROM students_temp WHERE id IN ($placeholders)");
$getTokensStmt->execute($student_ids);
$tokens = $getTokensStmt->fetchAll(PDO::FETCH_COLUMN);

if (!$tokens || count($tokens) === 0) {
    echo json_encode(['status'=>'error','message'=>'Selected students not found']);
    exit;
}
if (count($tokens) > 1) {
    echo json_encode(['status'=>'error','message'=>'Selected students belong to different schools — cannot bulk assign together']);
    exit;
}
$schoolToken = $tokens[0];

// ========== capacity check ===========
if ($section_id) {
    $capStmt = $pdo->prepare("SELECT capacity FROM sections WHERE id = ? LIMIT 1");
    $capStmt->execute([$section_id]);
} else {
    $capStmt = $pdo->prepare("SELECT capacity FROM classes WHERE id = ? LIMIT 1");
    $capStmt->execute([$class_id]);
}
$capacity = (int)$capStmt->fetchColumn();

if ($section_id) {
    $countStmt = $pdo->prepare("SELECT COUNT(*) FROM students_temp WHERE section_id = ? AND school_token = ?");
    $countStmt->execute([$section_id, $schoolToken]);
} else {
    $countStmt = $pdo->prepare("SELECT COUNT(*) FROM students_temp WHERE class_id = ? AND (section_id IS NULL OR section_id = 0) AND school_token = ?");
    $countStmt->execute([$class_id, $schoolToken]);
}
$existing = (int)$countStmt->fetchColumn();

if ($existing + $selectedCnt > $capacity) {
    echo json_encode([
        'status'  => 'error',
        'message' => "Cannot assign. Max capacity is $capacity, already $existing assigned."
    ]);
    exit;
}

if ($section_id) {
    $rollStmt = $pdo->prepare("SELECT COALESCE(MAX(roll_no), 0) FROM students_temp WHERE department_id = ? AND class_id = ? AND section_id = ? AND school_token = ?");
    $rollStmt->execute([$dept_id, $class_id, $section_id, $schoolToken]);
} else {
    $rollStmt = $pdo->prepare("SELECT COALESCE(MAX(roll_no), 0) FROM students_temp WHERE department_id = ? AND class_id = ? AND (section_id IS NULL OR section_id = 0) AND school_token = ?");
    $rollStmt->execute([$dept_id, $class_id, $schoolToken]);
}
$lastRoll = (int)$rollStmt->fetchColumn();

$nameStmt = $pdo->prepare("SELECT id, name, student_uid FROM students_temp WHERE id IN ($placeholders)");
$nameStmt->execute($student_ids);
$toAssign = $nameStmt->fetchAll(PDO::FETCH_ASSOC);

if (!$toAssign || count($toAssign) === 0) {
    echo json_encode(['status'=>'error','message'=>'No valid students to assign']);
    exit;
}

usort($toAssign, function($a, $b){
    return strcasecmp($a['name'] ?? '', $b['name'] ?? '');
});

$updStmt = $pdo->prepare("UPDATE students_temp SET department_id = :d, class_id = :c, section_id = :s, status = 'Assigned', roll_no = :r, student_qr = :qr WHERE id = :id");
$insertAssigned = $pdo->prepare("INSERT INTO student_assigned_class (student_id, dept_id, class_id, section_id) VALUES (:id, :d, :c, :s)");

$uploadDirRel = '/Admin/uploads/student/';       
$uploadDirSys = __DIR__ . '/uploads/student/';     

if (!is_dir($uploadDirSys)) {
    if (!mkdir($uploadDirSys, 0755, true)) {
        echo json_encode(['status'=>'error','message'=>'Unable to create upload directory']);
        exit;
    }
}

$errors = [];
$processed = 0;

// process each student in alphabetical order
foreach ($toAssign as $stu) {
    $sid = (int)$stu['id'];
    $studentUid = trim($stu['student_uid'] ?? '');

    $lastRoll++;
    $assignedRoll = $lastRoll;

    if ($studentUid === '') {
        $errors[] = "Missing student UID for ID $sid";
    }

    $qrPathDb = null;
    if ($studentUid !== '') {
        $apiUrl = 'https://api.qrserver.com/v1/create-qr-code/?data=' . urlencode($studentUid) . '&size=200x200';
        $imgData = @file_get_contents($apiUrl);
        if ($imgData === false) {
            $ch = curl_init($apiUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 20);
            $imgData = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            if ($imgData === false || ($httpCode !== 200 && $httpCode !== 0)) {
                $errors[] = "Failed to generate QR for UID {$studentUid} (ID $sid)";
                $imgData = false;
            }
        }
        if ($imgData !== false) {
            $safeName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $studentUid);
            $filename = $safeName . '_' . time() . '_' . random_int(1000,9999) . '.png';
            $savedPathSys = $uploadDirSys . $filename;
            $savedPathDb  = $uploadDirRel . $filename;

            if (file_put_contents($savedPathSys, $imgData) !== false) {
                $qrPathDb = $savedPathDb;
            } else {
                $errors[] = "Failed to save QR image for UID {$studentUid} (ID $sid)";
                $qrPathDb = null;
            }
        }
    }

    // update DB for this student
    try {
        $sVal = $section_id ?: null;
        $updStmt->execute([
            ':d' => $dept_id,
            ':c' => $class_id,
            ':s' => $sVal,
            ':r' => $assignedRoll,
            ':qr' => $qrPathDb,
            ':id' => $sid
        ]);
        try {
            $insertAssigned->execute([
                ':id' => $sid,
                ':d' => $dept_id,
                ':c' => $class_id,
                ':s' => $sVal
            ]);
        } catch (PDOException $iex) {
            error_log("student_assigned_class insert error for student $sid: " . $iex->getMessage());
        }
        $processed++;
    } catch (PDOException $ex) {
        if (!empty($savedPathSys) && file_exists($savedPathSys)) {
            @unlink($savedPathSys);
        }
        $errors[] = "DB error for student ID $sid: " . $ex->getMessage();
        continue;
    }
}

if ($processed === 0) {
    $msg = "No students were processed.";
    if ($errors) $msg .= " Errors: " . implode(' ; ', array_slice($errors, 0, 6));
    echo json_encode(['status'=>'error','message'=>$msg]);
    exit;
}

$msg = "$processed student(s) assigned with roll numbers and QR generated.";
if (!empty($errors)) $msg .= " Some issues: " . implode(' ; ', array_slice($errors, 0, 6));

echo json_encode(['status'=>'success','message'=>$msg]);
exit; ?>