<?php
header('Content-Type: application/json');
require 'config.php';
session_start();

try {
    if (empty($_SESSION['id'])) throw new Exception('Unauthorized');

    $student_id = isset($_POST['student_id']) ? (int)$_POST['student_id'] : 0;
    $book_ids = $_POST['book_ids'] ?? [];
    $supposed_submit_date = trim($_POST['supposed_submit_date'] ?? '');

    if (!$student_id) throw new Exception('Invalid student id');
    if (empty($book_ids) || !is_array($book_ids)) throw new Exception('No books selected');
    if (!$supposed_submit_date) throw new Exception('Select return date & time');

    $supposed_ts = date('Y-m-d H:i:s', strtotime($supposed_submit_date));
    if (!$supposed_ts) throw new Exception('Invalid date/time');

    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $failed = [];
    $assigned = [];

    foreach ($book_ids as $bidRaw) {
        $bid = (int)$bidRaw;
        if (!$bid) { $failed[] = "Invalid book id: {$bidRaw}"; continue; }

        // check if this student already has this *book title* assigned and not returned
        $check = $pdo->prepare("SELECT 1 FROM student_assigned_books sab JOIN book_copies bc ON sab.book_id = bc.copy_code
            WHERE sab.student_id = :sid AND bc.book_id = :bid AND (sab.actual_submit_date IS NULL OR sab.actual_submit_date = '') LIMIT 1");
        $check->execute([':sid' => $student_id, ':bid' => $bid]);
        if ($check->fetchColumn()) {
            $failed[] = "Student already has book id {$bid} assigned (active).";
            continue;
        }

        try {
            $pdo->beginTransaction();

            // Select one available copy for this book and lock it
            $sel = $pdo->prepare("SELECT id, copy_code FROM book_copies WHERE book_id = :bid AND status = 'available' LIMIT 1 FOR UPDATE");
            $sel->execute([':bid' => $bid]);
            $copy = $sel->fetch(PDO::FETCH_ASSOC);

            if (!$copy) {
                $pdo->rollBack();
                $failed[] = "No available copy for book id {$bid}";
                continue;
            }

            $copy_id = (int)$copy['id'];
            $copy_code = $copy['copy_code'];

            $ins = $pdo->prepare("INSERT INTO student_assigned_books (student_id, book_id, pickup_date, supposed_submit_date) VALUES (:sid, :copy_code, NOW(), :supp)");
            $ins->execute([':sid' => $student_id, ':copy_code' => $copy_code, ':supp' => $supposed_ts]);

            $upd = $pdo->prepare("UPDATE book_copies SET status = 'issued' WHERE id = :cid");
            $upd->execute([':cid' => $copy_id]);

            $pdo->commit();

            $assigned[] = ['book_id' => $bid, 'copy_id' => $copy_id, 'copy_code' => $copy_code];
        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            error_log("Assign book error for book {$bid}: " . $e->getMessage());
            $failed[] = "Failed to assign book id {$bid}";
        }
    }

    if (empty($assigned)) {
        throw new Exception('No book assigned. Details: ' . implode('; ', $failed));
    }

    $excludedBookIds = [];
    $sqlEx = "SELECT DISTINCT bc.book_id FROM student_assigned_books sab JOIN book_copies bc ON sab.book_id = bc.copy_code 
        WHERE sab.student_id = :sid AND (sab.actual_submit_date IS NULL OR sab.actual_submit_date = '')";
    $sEx = $pdo->prepare($sqlEx);
    $sEx->execute([':sid' => $student_id]);
    foreach ($sEx->fetchAll(PDO::FETCH_ASSOC) as $r) $excludedBookIds[] = (int)$r['book_id'];

    $params = [];
    $whereNotIn = '';
    if (!empty($excludedBookIds)) {
        $placeholders = implode(',', array_fill(0, count($excludedBookIds), '?'));
        $whereNotIn = " AND b.id NOT IN ($placeholders) ";
        $params = $excludedBookIds;
    }
    $sqlAvail = "SELECT b.id, b.name, b.des, b.author, b.image, COUNT(bc.id) AS available_copies FROM books b JOIN book_copies bc 
        ON bc.book_id = b.id AND bc.status = 'available' WHERE 1=1 $whereNotIn GROUP BY b.id HAVING available_copies > 0 ORDER BY b.name";
    $sAvail = $pdo->prepare($sqlAvail);
    $sAvail->execute($params);
    $available_books = $sAvail->fetchAll(PDO::FETCH_ASSOC);

    // assigned rows for this student
    $sqlAssigned = "SELECT sab.id AS assigned_id, sab.book_id AS copy_code, sab.pickup_date, sab.supposed_submit_date, sab.actual_submit_date, 
        bc.book_id AS book_id, b.name AS book_name, b.author, b.image FROM student_assigned_books sab JOIN book_copies bc ON sab.book_id = bc.copy_code 
        JOIN books b ON bc.book_id = b.id WHERE sab.student_id = :sid ORDER BY sab.pickup_date DESC";
    $sA = $pdo->prepare($sqlAssigned);
    $sA->execute([':sid' => $student_id]);
    $assignedRows = $sA->fetchAll(PDO::FETCH_ASSOC);

    $msg = count($assigned) . ' book(s) assigned successfully.';
    if (!empty($failed)) $msg .= ' Some failed: ' . implode('; ', $failed);

    echo json_encode(['status' => 'success', 'message' => $msg, 'assigned' => $assigned, 'failed' => $failed, 'available_books' => $available_books, 'assigned_rows' => $assignedRows]);
    exit;

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    exit;
} ?>