<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Access-Control-Allow-Headers, Content-Type, Access-Control-Allow-Methods, Authorization, X-Requested-With');

include "configDealer.php"; // this should define $conn (mysqli)

if (!$conn) {
    echo json_encode([
        'status' => false,
        'message' => "Database connection failed."
    ]);
    exit();
}

try {
    if (empty($_POST['teacher_id'])) {
        echo json_encode([
            'status' => false,
            'message' => "teacher id required."
        ]);
        mysqli_close($conn);
        exit();
    }

    $book_id = intval($_POST['book_id'] ?? 0);
    $add_qty = intval($_POST['add_qty'] ?? 0);

    if ($book_id < 1 || $add_qty < 1) {
        echo json_encode([
            'status' => false,
            'message' => "Book ID and quantity required."
        ]);
        mysqli_close($conn);
        exit();
    }

    $qrDir = '../Admin/uploads/book_qr/';
    if (!is_dir($qrDir) && !mkdir($qrDir, 0777, true)) {
        echo json_encode([
            'status' => false,
            'message' => "Failed to create QR directory."
        ]);
        mysqli_close($conn);
        exit();
    }

    // Start transaction
    mysqli_begin_transaction($conn);

    // Lock row for update
    $stmt = $conn->prepare("SELECT stock FROM books WHERE id = ? FOR UPDATE");
    $stmt->bind_param("i", $book_id);
    $stmt->execute();
    $stmt->bind_result($current_stock);
    if (!$stmt->fetch()) {
        mysqli_rollback($conn);
        throw new Exception("Book not found.");
    }
    $stmt->close();

    $stmt = $conn->prepare("SELECT COALESCE(MAX(CAST(SUBSTRING_INDEX(copy_code, '-', -1) AS UNSIGNED)), 0) AS lastnum FROM book_copies WHERE book_id = ?");
    $stmt->bind_param("i", $book_id);
    $stmt->execute();
    $stmt->bind_result($lastnum);
    $stmt->fetch();
    $stmt->close();

    $new_stock_total = $current_stock + $add_qty;

    $upd = $conn->prepare("UPDATE books SET stock = ? WHERE id = ?");
    $upd->bind_param("ii", $new_stock_total, $book_id);
    $upd->execute();
    $upd->close();

    $ins = $conn->prepare("INSERT INTO book_copies (book_id, copy_code, code_type, code_image, status) VALUES (?, ?, 'QR', ?, 'available')");

    for ($i = 1; $i <= $add_qty; $i++) {
        $seq = $lastnum + $i;
        $copyCode = $book_id . '-' . str_pad($seq, 4, '0', STR_PAD_LEFT);
        $qrFile = "qr_{$copyCode}.png";
        $qrPath = $qrDir . $qrFile;
        $apiUrl = 'https://api.qrserver.com/v1/create-qr-code/' . '?data=' . urlencode($copyCode) . '&size=200x200';

        $ch = curl_init($apiUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT        => 20,
        ]);
        $qrData = curl_exec($ch);
        $err    = curl_error($ch);
        curl_close($ch);

        if ($err || !$qrData) {
            mysqli_rollback($conn);
            throw new Exception("Failed to generate QR for {$copyCode}: {$err}");
        }

        if (file_put_contents($qrPath, $qrData) === false) {
            mysqli_rollback($conn);
            throw new Exception("Failed to save QR file: {$qrPath}");
        }

        $imgPath = 'uploads/book_qr/' . $qrFile;
        $ins->bind_param("iss", $book_id, $copyCode, $imgPath);
        $ins->execute();
    }
    $ins->close();

    mysqli_commit($conn);

    echo json_encode([
        'status' => true,
        'message' => "Stock updated: +{$add_qty} copies added. New total: {$new_stock_total}."
    ]);
    mysqli_close($conn);
    exit;

} catch (Exception $e) {
    if ($conn->connect_errno === 0 && mysqli_errno($conn) === 0) {
        mysqli_rollback($conn);
    }
    error_log("update_stock.php Error: " . $e->getMessage());
    echo json_encode([
        'status' => false,
        'message' => $e->getMessage()
    ]);
    mysqli_close($conn);
    exit;
}
?>